#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion
namespace gov.va.med.vbecs.DAL.UnitTests
{
	[TestFixture]
	public class DAL_PatientTransfusion : BaseTest 
	{
		
		private DataTable _dtPatientTransfusion;
		private DataTable _dtBloodUnit;

		[SetUp]
		protected void SetUp() 
		{				
			this.BaseSetUp();

			if (this.RefreshData)
			{
				new BOL.UnitTests.BOL_PatientTransfusionReaction().SaveImplicatedBloodUnits_Pass();

				//**PatientTransfusion
				this._dtPatientTransfusion = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM PatientTransfusion ORDER BY LastUpdateDate DESC", false);

				//**BloodUnit
				this._dtBloodUnit = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM vw_BloodShipAllCurrent WHERE BloodUnitGuid = '" + this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.BloodUnitGuid].ToString() + "'", true);
				
				this.RefreshData = false;
			}

		}

		[Test]
		public void Save_Pass()
		{
			this.RefreshData = true;

			System.Guid ptGuid = System.Guid.NewGuid();
			System.Guid patientGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid];
			System.Guid unitGuid = (System.Guid)this._dtBloodUnit.Rows[0][TABLES.BloodUnitStatus.BloodUnitGuid];
			byte[] rowVersion = (byte[])this._dtBloodUnit.Rows[0][ARTIFICIAL.UnitStatusRowVersion];

			DataTable dtITransfusion = SprocHelper.GetInsertPatientTransfusionSprocTable(ptGuid, patientGuid, unitGuid);
			DataTable dtIWorkload = new DataTable();

			bool retValue = DAL.PatientTransfusion.Save(dtITransfusion, dtIWorkload, null, null);

			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		[ExpectedException(typeof(Common.UniqueDatabaseConstraintViolatedException))]
		public void Save_Fail()
		{
			System.Guid ptGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientTransfusionGuid];
			System.Guid patientGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid];
			System.Guid unitGuid = (System.Guid)this._dtBloodUnit.Rows[0][TABLES.BloodUnitStatus.BloodUnitGuid];
	
			DataTable dtITransfusion = SprocHelper.GetInsertPatientTransfusionSprocTable(ptGuid, patientGuid, unitGuid);
			
			DataTable dtIWorkload = new DataTable();
			DAL.PatientTransfusion.Save(dtITransfusion, dtIWorkload, null, null);
		}

		[Test]
		public void Update_Pass()
		{
			this.RefreshData = true;

			byte[] rowVersion = (byte[])this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.RowVersion];
			System.Guid transfusionGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientTransfusionGuid];
			System.Guid patientGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid];
			System.Guid unitGuid = (System.Guid)this._dtBloodUnit.Rows[0][TABLES.BloodUnitStatus.BloodUnitGuid];

			DataTable dtUTransfusion = SprocHelper.GetUpdatePatientTransfusionRecordSprocTable(transfusionGuid, rowVersion, patientGuid, unitGuid);

			bool retValue = DAL.PatientTransfusion.Update(dtUTransfusion, null, false, null, null);

			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		[ExpectedException(typeof(Common.RowVersionException))]
		public void Update_Fail()
		{
			byte[] rowVersion = new byte[8];
			System.Guid transfusionGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientTransfusionGuid];
			System.Guid patientGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid];
			System.Guid unitGuid = (System.Guid)this._dtBloodUnit.Rows[0][TABLES.BloodUnitStatus.BloodUnitGuid];

			DataTable dtUTransfusion = SprocHelper.GetUpdatePatientTransfusionRecordSprocTable(transfusionGuid, rowVersion, patientGuid, unitGuid);

			DAL.PatientTransfusion.Update(dtUTransfusion, null, false, null, null);
		}

		[Test]
		public void GetTransfusionComment_Pass()
		{
			System.Guid transfusionGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.BloodUnitGuid];

			DataTable dtTest = DAL.PatientTransfusion.GetTransfusionComment(transfusionGuid);

			Assert.AreEqual(11, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusionComment.PatientTransfusionCommentGuid), TABLES.PatientTransfusionComment.PatientTransfusionCommentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusionComment.PatientTransfusionGuid), TABLES.PatientTransfusionComment.PatientTransfusionGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusionComment.PatientTransfusionReactionGuid), TABLES.PatientTransfusionComment.PatientTransfusionReactionGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusionComment.CommentCategoryCode), TABLES.PatientTransfusionComment.CommentCategoryCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusionComment.CannedCommentGuid), TABLES.PatientTransfusionComment.CannedCommentGuid);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusionComment.PatientTransfusionCommentText), TABLES.PatientTransfusionComment.PatientTransfusionCommentText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusionComment.DivisionCode), TABLES.PatientTransfusionComment.DivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusionComment.RecordStatusCode), TABLES.PatientTransfusionComment.RecordStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusionComment.LastUpdateUser), TABLES.PatientTransfusionComment.LastUpdateUser);	
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusionComment.LastUpdateFunctionId), TABLES.PatientTransfusionComment.LastUpdateFunctionId);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusionComment.RowVersion), TABLES.PatientTransfusionComment.RowVersion);

		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void GetTransfusionComment_Fail()
		{
			DAL.PatientTransfusion.GetTransfusionComment(Guid.Empty);
		}

		[Test]
		public void GetIssuedAndTransfusedUnitsForPatient_Pass()
		{
			System.Guid patientGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid];

			DataTable dtTest = DAL.PatientTransfusion.GetIssuedAndTransfusedUnitsForPatient(patientGuid);

			Assert.AreEqual(22, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.IssuedUnit.IssuedUnitGuid), TABLES.IssuedUnit.IssuedUnitGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.IssuedUnit.IssueTechId), TABLES.IssuedUnit.IssueTechId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.IssuedUnit.RemoteStorageIndicator), TABLES.IssuedUnit.RemoteStorageIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.BloodUnitGuid), TABLES.BloodUnit.BloodUnitGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitStatus.BloodUnitStatusGuid), TABLES.BloodUnitStatus.BloodUnitStatusGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.EyeReadableUnitId), TABLES.BloodUnit.EyeReadableUnitId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodProduct.ProductShortName), TABLES.BloodProduct.ProductShortName);			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.ProductType.ProductTypeName), TABLES.ProductType.ProductTypeName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.RhFactor.RhFactorText), TABLES.RhFactor.RhFactorText);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.ABORh), ARTIFICIAL.ABORh);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitStatusCodeCurrent.UnitStatusCode), TABLES.BloodUnitStatusCodeCurrent.UnitStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.OriginalVolume), TABLES.BloodUnit.OriginalVolume);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.UnitProductCode), TABLES.BloodUnit.UnitProductCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitStatus.PresumedTransfusedIndicator), TABLES.BloodUnitStatus.PresumedTransfusedIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.ComponentClassParameter.MaximumTransfusionTime), TABLES.ComponentClassParameter.MaximumTransfusionTime);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitMedia.UnitExpirationDate), TABLES.BloodUnitMedia.UnitExpirationDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.IssuedUnit.IssueDateTime), TABLES.IssuedUnit.IssueDateTime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionEndDateTime), TABLES.PatientTransfusion.TransfusionEndDateTime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.UnitDivisionId), TABLES.BloodUnit.UnitDivisionId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.LabelTypeCode), TABLES.BloodUnit.LabelTypeCode);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.DonationType.DonationTypeCode), TABLES.DonationType.DonationTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.ApplicationOfRecordId), TABLES.PatientTransfusion.ApplicationOfRecordId);
			
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void GetIssuedAndTransfusedUnitsForPatient_Fail()
		{
			DataTable dtTest = DAL.PatientTransfusion.GetIssuedAndTransfusedUnitsForPatient(Guid.Empty);
		}

		[Test]
		public void GetPatientTransfusionRecord_Guid_Guid_Pass()
		{
			System.Guid patientGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid];
			System.Guid unitGuid = (System.Guid)this._dtBloodUnit.Rows[0][TABLES.BloodUnitStatus.BloodUnitGuid];

			DataTable dtTest = DAL.PatientTransfusion.GetPatientTransfusionRecord(patientGuid, unitGuid);

			Assert.AreEqual(21, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.PatientTransfusionGuid), TABLES.PatientTransfusion.PatientTransfusionGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.PatientGuid), TABLES.PatientTransfusion.PatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionLocation), TABLES.PatientTransfusion.TransfusionLocation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.LocationIen), TABLES.PatientTransfusion.LocationIen);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionStartDateTime), TABLES.PatientTransfusion.TransfusionStartDateTime);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionEndDateTime), TABLES.PatientTransfusion.TransfusionEndDateTime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.BedsideVerificationDate), TABLES.PatientTransfusion.BedsideVerificationDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.BloodUnitGuid), TABLES.PatientTransfusion.BloodUnitGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.Transfusionist1Id), TABLES.PatientTransfusion.Transfusionist1Id);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.Transfusionist2Id), TABLES.PatientTransfusion.Transfusionist2Id);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusedVolume), TABLES.PatientTransfusion.TransfusedVolume);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.InterruptionIndicator), TABLES.PatientTransfusion.InterruptionIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.ReactionIndicator), TABLES.PatientTransfusion.ReactionIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.ReactionSymptoms), TABLES.PatientTransfusion.ReactionSymptoms);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.DivisionCode), TABLES.PatientTransfusion.DivisionCode);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.LastUpdateUser), TABLES.PatientTransfusion.LastUpdateUser);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.LastUpdateFunctionId), TABLES.PatientTransfusion.LastUpdateFunctionId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.RowVersion), TABLES.PatientTransfusion.RowVersion);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.ApplicationOfRecordId), TABLES.PatientTransfusion.ApplicationOfRecordId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.InterruptionComment), TABLES.PatientTransfusion.InterruptionComment);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.BedsideVerificationUnitManualEntryReason), TABLES.PatientTransfusion.BedsideVerificationUnitManualEntryReason);
		}

		[Test]
		public void GetPatientTransfusionRecord_Guid_Guid_Fail()
		{
			DataTable dtTest = DAL.PatientTransfusion.GetPatientTransfusionRecord(Guid.NewGuid(), Guid.NewGuid());
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetPatientTransfusionRecord_Guid_Pass()
		{
			System.Guid unitGuid = (System.Guid)this._dtBloodUnit.Rows[0][TABLES.BloodUnitStatus.BloodUnitGuid];


			DataTable dtTest = DAL.PatientTransfusion.GetPatientTransfusionRecord(unitGuid);
			Assert.AreEqual(21, dtTest.Columns.Count, "Columns.Count");
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.PatientTransfusionGuid), TABLES.PatientTransfusion.PatientTransfusionGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.PatientGuid), TABLES.PatientTransfusion.PatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionLocation), TABLES.PatientTransfusion.TransfusionLocation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.LocationIen), TABLES.PatientTransfusion.LocationIen);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionStartDateTime), TABLES.PatientTransfusion.TransfusionStartDateTime);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionEndDateTime), TABLES.PatientTransfusion.TransfusionEndDateTime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.BedsideVerificationDate), TABLES.PatientTransfusion.BedsideVerificationDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.BloodUnitGuid), TABLES.PatientTransfusion.BloodUnitGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.Transfusionist1Id), TABLES.PatientTransfusion.Transfusionist1Id);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.Transfusionist2Id), TABLES.PatientTransfusion.Transfusionist2Id);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusedVolume), TABLES.PatientTransfusion.TransfusedVolume);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.InterruptionIndicator), TABLES.PatientTransfusion.InterruptionIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.ReactionIndicator), TABLES.PatientTransfusion.ReactionIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.ReactionSymptoms), TABLES.PatientTransfusion.ReactionSymptoms);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.DivisionCode), TABLES.PatientTransfusion.DivisionCode);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.LastUpdateUser), TABLES.PatientTransfusion.LastUpdateUser);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.LastUpdateFunctionId), TABLES.PatientTransfusion.LastUpdateFunctionId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.RowVersion), TABLES.PatientTransfusion.RowVersion);
		
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.ApplicationOfRecordId), TABLES.PatientTransfusion.ApplicationOfRecordId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.InterruptionComment), TABLES.PatientTransfusion.InterruptionComment);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.BedsideVerificationUnitManualEntryReason), TABLES.PatientTransfusion.BedsideVerificationUnitManualEntryReason);
		}

		[Test]
		public void GetPatientTransfusionRecord_Guid_Fail()
		{
			DataTable dtTest = DAL.PatientTransfusion.GetPatientTransfusionRecord(Guid.NewGuid());
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetRecentlyIssuedAndTransfusedUnitsForPatient_Pass()
		{
			System.Guid patientGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid];

			DataTable dtTest = DAL.PatientTransfusion.GetRecentlyIssuedAndTransfusedUnitsForPatient(patientGuid);

			Assert.AreEqual(19, dtTest.Columns.Count, "Columns.Count");
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.EyeReadableUnitId), TABLES.BloodUnit.EyeReadableUnitId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodProduct.ProductShortName), TABLES.BloodProduct.ProductShortName);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.PooledUnitsCount), ARTIFICIAL.PooledUnitsCount);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitMedia.BloodTypeCode), TABLES.BloodUnitMedia.BloodTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitMedia.RhFactorCode), TABLES.BloodUnitMedia.RhFactorCode);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitMedia.UnitExpirationDate), TABLES.BloodUnitMedia.UnitExpirationDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.DonationType.DonationTypeCode), TABLES.DonationType.DonationTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.DonationType.DonationTypeText), TABLES.DonationType.DonationTypeText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitStatusCodeCurrent.UnitStatusCode), TABLES.BloodUnitStatusCodeCurrent.UnitStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.UnitStatus.UnitStatusText), TABLES.UnitStatus.UnitStatusText);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitStatus.PresumedTransfusedIndicator), TABLES.BloodUnitStatus.PresumedTransfusedIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionEndDateTime), TABLES.PatientTransfusion.TransfusionEndDateTime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TransfusionReactionType.TransfusionReactionTypeText), TABLES.TransfusionReactionType.TransfusionReactionTypeText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.DivisionCode), TABLES.PatientTransfusion.DivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusionComment.CannedCommentGuid), TABLES.PatientTransfusionComment.CannedCommentGuid);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitMedia.Comments), TABLES.BloodUnitMedia.Comments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.UnitDivisionId), TABLES.BloodUnit.UnitDivisionId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.LabelTypeCode), TABLES.BloodUnit.LabelTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.UnitProductCode), TABLES.BloodUnit.UnitProductCode);

		}

		[Test]
		public void GetRecentlyIssuedAndTransfusedUnitsForPatient_Fail()
		{
			DataTable dtTest = DAL.PatientTransfusion.GetRecentlyIssuedAndTransfusedUnitsForPatient(System.Guid.NewGuid());
			
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetRecentlyIssuedAndTransfusedUnitsForPatientCount_Pass()
		{
			Assert.IsTrue(DAL.PatientTransfusion.GetRecentlyIssuedAndTransfusedUnitsForPatientCount((System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid]) >= 0);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void GetRecentlyIssuedAndTransfusedUnitsForPatientCount_Fail()
		{
			Assert.IsTrue(DAL.PatientTransfusion.GetRecentlyIssuedAndTransfusedUnitsForPatientCount(Guid.Empty)==0);
		}

		[Test]
		public void IsTransfusionHistoryforPatient_Pass()
		{
			System.Guid patientGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid];

			bool val = DAL.PatientTransfusion.IsTransfusionHistoryforPatient(patientGuid);
			Assert.AreEqual(true, true, "ReturnValue");		//true or false is success
		}

		[Test]
		public void IsTransfusionHistoryforPatient_Fail()
		{
			bool retValue = DAL.PatientTransfusion.IsTransfusionHistoryforPatient(Guid.NewGuid());
			
			Assert.AreEqual(false, retValue, "ReturnValue");
		}

		[Test]
		public void GetTransfusionRelatedInformation_Pass()
		{
			new BOL.UnitTests.BOL_PatientTransfusion().GetIssuedAndTransfusedUnitsForPatient_Pass();
			
			DataTable dtTmp = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM PatientTransfusion WHERE BloodUnitGuid IN (SELECT BloodUnitGuid FROM IssuedUnit) ORDER BY LastUpdateDate DESC");

			System.Guid patientGuid = (System.Guid)dtTmp.Rows[0][TABLES.PatientTransfusion.PatientGuid];
			System.Guid unitGuid = (System.Guid)dtTmp.Rows[0][TABLES.PatientTransfusion.BloodUnitGuid];

			DataRow dtRow = DAL.PatientTransfusion.GetTransfusionRelatedInformation(patientGuid, unitGuid);

			Assert.AreEqual(5, dtRow.Table.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtRow.Table.Columns.Contains(TABLES.IssuedUnit.IssueDateTime), TABLES.IssuedUnit.IssueDateTime);
			Assert.AreEqual(true, dtRow.Table.Columns.Contains(TABLES.IssuedUnit.RemoteStorageIndicator), TABLES.IssuedUnit.RemoteStorageIndicator);
			Assert.AreEqual(true, dtRow.Table.Columns.Contains(TABLES.ComponentClass.ComponentClassId), TABLES.ComponentClass.ComponentClassId);
			Assert.AreEqual(true, dtRow.Table.Columns.Contains(TABLES.BloodUnit.OriginalVolume), TABLES.BloodUnit.OriginalVolume);
			Assert.AreEqual(true, dtRow.Table.Columns.Contains(TABLES.ComponentClassParameter.MaximumTransfusionTime), TABLES.ComponentClassParameter.MaximumTransfusionTime);
		}

		[Test]
		public void GetTransfusionRelatedInformation_Fail()
		{
			DataRow dtRow = DAL.PatientTransfusion.GetTransfusionRelatedInformation(Guid.NewGuid(), Guid.NewGuid());
			
			Assert.AreEqual(null, dtRow, "ReturnValue");
		}

		[Test]
		public void GetTransfusedPatient_Pass()
		{
			System.Guid unitGuid = (System.Guid)this._dtBloodUnit.Rows[0][TABLES.BloodUnitStatus.BloodUnitGuid];

			DataTable dtTest = DAL.PatientTransfusion.GetTransfusedPatient(unitGuid);

			Assert.AreEqual(5, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.PatientGuid), TABLES.PatientTransfusion.PatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientFirstName), TABLES.Patient.PatientFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientMiddleName), TABLES.Patient.PatientMiddleName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientLastName), TABLES.Patient.PatientLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.VistaPatientId), TABLES.Patient.VistaPatientId);

		}

		[Test]
		public void GetTransfusedPatient_Fail()
		{
			//TODO: Check for Tables.Count > 0 will always be met - the else block can be removed - unreachable
			DataTable dtTest = DAL.PatientTransfusion.GetTransfusedPatient(Guid.NewGuid());
			
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetTransfusionHistoryForPatient_Pass()
		{
			System.Guid patientGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid];
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now.AddYears(1);

			DataTable dtTest = DAL.PatientTransfusion.GetTransfusionHistoryForPatient(patientGuid, startDate, endDate);

			Assert.AreEqual(26, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.PatientTransfusionGuid), TABLES.PatientTransfusion.PatientTransfusionGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.PatientGuid), TABLES.PatientTransfusion.PatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.PatientSpecimenGuid), TABLES.PatientTransfusion.PatientSpecimenGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionLocation), TABLES.PatientTransfusion.TransfusionLocation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.LocationIen), TABLES.PatientTransfusion.LocationIen);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionStartDateTime), TABLES.PatientTransfusion.TransfusionStartDateTime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionEndDateTime), TABLES.PatientTransfusion.TransfusionEndDateTime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.BloodUnitGuid), TABLES.PatientTransfusion.BloodUnitGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.BedsideVerificationDate), TABLES.PatientTransfusion.BedsideVerificationDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.Transfusionist1Id), TABLES.PatientTransfusion.Transfusionist1Id);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.Transfusionist2Id), TABLES.PatientTransfusion.Transfusionist2Id);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusedVolume), TABLES.PatientTransfusion.TransfusedVolume);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionPooledUnitsCount), TABLES.PatientTransfusion.TransfusionPooledUnitsCount);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.PhlebotomistId), TABLES.PatientTransfusion.PhlebotomistId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.InterruptionIndicator), TABLES.PatientTransfusion.InterruptionIndicator);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.InterruptionDateTime), TABLES.PatientTransfusion.InterruptionDateTime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.InterruptionComment), TABLES.PatientTransfusion.InterruptionComment);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.ReactionIndicator), TABLES.PatientTransfusion.ReactionIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.ReactionSymptoms), TABLES.PatientTransfusion.ReactionSymptoms);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.OtherDifficultyIndicator), TABLES.PatientTransfusion.OtherDifficultyIndicator);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.OtherDifficultyText), TABLES.PatientTransfusion.OtherDifficultyText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.DivisionCode), TABLES.PatientTransfusion.DivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.RecordStatusCode), TABLES.PatientTransfusion.RecordStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.LastUpdateDate), TABLES.PatientTransfusion.LastUpdateDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.LastUpdateUser), TABLES.PatientTransfusion.LastUpdateUser);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.RowVersion), TABLES.PatientTransfusion.RowVersion);
		}

		[Test]
		public void GetTransfusionHistoryForPatient_Fail()
		{
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now.AddYears(1);

			DataTable dtTest = DAL.PatientTransfusion.GetTransfusionHistoryForPatient(Guid.NewGuid(), startDate, endDate);
			
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetTransfusionHistoryForPatientSpecimen_Pass()
		{
			DataTable dtTest = DAL.PatientTransfusion.GetTransfusionHistoryForPatientSpecimen(System.Guid.NewGuid());

			Assert.AreEqual(3, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.BloodUnitGuid), TABLES.PatientTransfusion.BloodUnitGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.PatientGuid), TABLES.PatientTransfusion.PatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitStatus.PresumedTransfusedIndicator), TABLES.BloodUnitStatus.PresumedTransfusedIndicator);

		}
		
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void GetTransfusionHistoryForPatientSpecimen_Fail()
		{
			//TODO: Check for Tables.Count > 0 will always be met - the else block can be removed - unreachable
			
			DAL.PatientTransfusion.GetTransfusionHistoryForPatientSpecimen(Guid.Empty);
		}


		[Test]
		public void GetTransfusedUnitsForPatient_Pass()
		{
			System.Guid patientGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid];

			DataTable dtTest = DAL.PatientTransfusion.GetTransfusedUnitsForPatient(patientGuid, this.TestDivisionCode);

			Assert.AreEqual(46, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.BloodUnitGuid), TABLES.BloodUnit.BloodUnitGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.EyeReadableUnitId), TABLES.BloodUnit.EyeReadableUnitId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitMedia.DirectedPatientGuid), TABLES.BloodUnitMedia.DirectedPatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.UnitProductCode), TABLES.BloodUnit.UnitProductCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitMedia.RhFactorCode), TABLES.BloodUnitMedia.RhFactorCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitMedia.UnitExpirationDate), TABLES.BloodUnitMedia.UnitExpirationDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.DonationTypeId), TABLES.BloodUnit.DonationTypeId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.UnitInDate), TABLES.BloodUnit.UnitInDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.DivisionCode), TABLES.BloodUnit.DivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitMedia.BloodTypeCode), TABLES.BloodUnitMedia.BloodTypeCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.UnitFdaRegistrationNumber), TABLES.BloodUnit.UnitFdaRegistrationNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.LabelTypeCode), TABLES.BloodUnit.LabelTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.LabelUnitId), TABLES.BloodUnit.LabelUnitId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodProduct.ProductTypeCode), TABLES.BloodProduct.ProductTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.BloodProductGuid), TABLES.BloodUnit.BloodProductGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodProduct.ProductName), TABLES.BloodProduct.ProductName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodProduct.ProductShortName), TABLES.BloodProduct.ProductShortName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitStatus.BiohazardousWarningIndicator), TABLES.BloodUnitStatus.BiohazardousWarningIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.DonationType.DonationTypeText), TABLES.DonationType.DonationTypeText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.ProductType.ProductTypeName), TABLES.ProductType.ProductTypeName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.RhFactor.RhFactorText), TABLES.RhFactor.RhFactorText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Facility.FacilityName), TABLES.Facility.FacilityName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientFirstName), TABLES.Patient.PatientFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientMiddleName), TABLES.Patient.PatientMiddleName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientLastName), TABLES.Patient.PatientLastName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodProduct.MaximumStorageTime), TABLES.BloodProduct.MaximumStorageTime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.VistaPatientId), TABLES.Patient.VistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientSsn), TABLES.Patient.PatientSsn);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.DisplayVistaPatientId), TABLES.Patient.DisplayVistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitStatus.UnitStatusComments), TABLES.BloodUnitStatus.UnitStatusComments);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitStatus.DiscardCannedCommentGuid), TABLES.BloodUnitStatus.DiscardCannedCommentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitStatusCodeCurrent.UnitStatusCode), TABLES.BloodUnitStatusCodeCurrent.UnitStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.UnitStatus.UnitStatusText), TABLES.UnitStatus.UnitStatusText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitStatus.QuarantineIndicator), TABLES.BloodUnitStatus.QuarantineIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitMedia.TestingStatusId), TABLES.BloodUnitMedia.TestingStatusId);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.OriginalVolume), TABLES.BloodUnit.OriginalVolume);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.RowVersion), TABLES.BloodUnit.RowVersion);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.UnitStatusRowVersion), ARTIFICIAL.UnitStatusRowVersion);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitMedia.Comments), TABLES.BloodUnitMedia.Comments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitShipment.IncomingShipmentGuid), TABLES.BloodUnitShipment.IncomingShipmentGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.ABORh), ARTIFICIAL.ABORh);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.UnitDivisionId), TABLES.BloodUnit.UnitDivisionId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentDate), TABLES.Shipment.ShipmentDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitStatus.PresumedTransfusedIndicator), TABLES.BloodUnitStatus.PresumedTransfusedIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.FullProductCode), ARTIFICIAL.FullProductCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.PatientTransfusionGuid), TABLES.PatientTransfusion.PatientTransfusionGuid);
		}
		
		[Test]
		public void GetTransfusedUnitsForPatient_Fail()
		{
			try
			{
				DataTable dtTest = DAL.PatientTransfusion.GetTransfusedUnitsForPatient(Guid.Empty, this.TestDivisionCode);
				Assert.IsTrue(false, "1-" + AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}

			try
			{
				DataTable dtTest = DAL.PatientTransfusion.GetTransfusedUnitsForPatient(Guid.NewGuid(), string.Empty);
				Assert.IsTrue(false, "2-" + AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
		}

		[Test]
		public void GetPatientTransfusionRecords_Pass()
		{
			System.Guid patientGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid];

			DataTable dtTest = DAL.PatientTransfusion.GetPatientTransfusionRecords(patientGuid, this.TestDivisionCode);

			Assert.AreEqual(17, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.PatientTransfusionGuid), TABLES.PatientTransfusion.PatientTransfusionGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.PatientGuid), TABLES.PatientTransfusion.PatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionLocation), TABLES.PatientTransfusion.TransfusionLocation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.LocationIen), TABLES.PatientTransfusion.LocationIen);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionStartDateTime), TABLES.PatientTransfusion.TransfusionStartDateTime);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionEndDateTime), TABLES.PatientTransfusion.TransfusionEndDateTime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.BloodUnitGuid), TABLES.PatientTransfusion.BloodUnitGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.Transfusionist1Id), TABLES.PatientTransfusion.Transfusionist1Id);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.Transfusionist2Id), TABLES.PatientTransfusion.Transfusionist2Id);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusedVolume), TABLES.PatientTransfusion.TransfusedVolume);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.InterruptionIndicator), TABLES.PatientTransfusion.InterruptionIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.ReactionIndicator), TABLES.PatientTransfusion.ReactionIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.ReactionSymptoms), TABLES.PatientTransfusion.ReactionSymptoms);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.DivisionCode), TABLES.PatientTransfusion.DivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.LastUpdateUser), TABLES.PatientTransfusion.LastUpdateUser);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.LastUpdateFunctionId), TABLES.PatientTransfusion.LastUpdateFunctionId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.RowVersion), TABLES.PatientTransfusion.RowVersion);
		}
		
		[Test]
		public void GetPatientTransfusionRecords_Fail()
		{
			try
			{
				DataTable dtTest = DAL.PatientTransfusion.GetPatientTransfusionRecords(Guid.Empty, this.TestDivisionCode);
				Assert.IsTrue(false, "1-" + AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}

			try
			{
				DataTable dtTest = DAL.PatientTransfusion.GetPatientTransfusionRecords(Guid.NewGuid(), string.Empty);
				Assert.IsTrue(false, "2-" + AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
		}

		[Test]
		public void GetTransfusedUnitsInDateRangeCount_Pass()
		{
			System.Guid patientGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid];
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now.AddYears(1);

			int retCount = DAL.PatientTransfusion.GetTransfusedUnitsInDateRangeCount(startDate, endDate, this.TestDivisionCode, patientGuid);

			Assert.AreEqual(true, retCount >= 0, "ReturnValue");
		}

		[Test]
		public void GetTransfusedUnitsInDateRangeCount_Fail()
		{
			System.Guid patientGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid];
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now.AddYears(1);

			int retCount = DAL.PatientTransfusion.GetTransfusedUnitsInDateRangeCount(startDate, endDate, string.Empty, patientGuid);

			Assert.AreEqual(0, retCount, "ReturnValue");
		}

		[Test]
		public void GetTransfusedUnitsInDateRange_Pass()
		{
			System.Guid patientGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid];
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now.AddYears(1);

			DataTable dtTest = DAL.PatientTransfusion.GetTransfusedUnitsInDateRange(startDate, endDate, this.TestDivisionCode, patientGuid);

			Assert.AreEqual(16, dtTest.Columns.Count, "Columns.Count");
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientGuid), TABLES.Patient.PatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientLastName), TABLES.Patient.PatientLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientFirstName), TABLES.Patient.PatientFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientMiddleName), TABLES.Patient.PatientMiddleName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.DisplayVistaPatientId), TABLES.Patient.DisplayVistaPatientId);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientDeathDate), TABLES.Patient.PatientDeathDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientLocation), TABLES.PatientTreatment.PatientLocation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.VistaPatientId), TABLES.Patient.VistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.EyeReadableUnitId), TABLES.BloodUnit.EyeReadableUnitId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodProduct.ProductShortName), TABLES.BloodProduct.ProductShortName);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.PooledUnitsCount), ARTIFICIAL.PooledUnitsCount);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.TransfusionEndDateTime), TABLES.PatientTransfusion.TransfusionEndDateTime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.LabelTypeCode), TABLES.BloodUnit.LabelTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.UnitDivisionId), TABLES.BloodUnit.UnitDivisionId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.DonationType.DonationTypeCode), TABLES.DonationType.DonationTypeCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.UnitProductCode), TABLES.BloodUnit.UnitProductCode);
		}

		[Test]
		public void GetTransfusedUnitsInDateRange_Fail()
		{
			System.Guid patientGuid = (System.Guid)this._dtPatientTransfusion.Rows[0][TABLES.PatientTransfusion.PatientGuid];
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now.AddYears(1);

			DataTable dtTest = DAL.PatientTransfusion.GetTransfusedUnitsInDateRange(startDate, endDate, string.Empty, patientGuid);

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetTransfusionReport_Pass()
		{
			string filter = "";
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now.AddYears(1);

			DataTable dtTest = DAL.PatientTransfusion.GetTransfusionReport(startDate, endDate, filter, this.TestDivisionCode,1);

			Assert.AreEqual(17, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientGuid), TABLES.Patient.PatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.DisplayVistaPatientId), TABLES.Patient.DisplayVistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientLastName), TABLES.Patient.PatientLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientFirstName), TABLES.Patient.PatientFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.EyeReadableUnitId), TABLES.BloodUnit.EyeReadableUnitId);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodProduct.ProductShortName), TABLES.BloodProduct.ProductShortName);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.PooledUnitsCount), ARTIFICIAL.PooledUnitsCount);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.IssuedUnit.IssueDateTime), TABLES.IssuedUnit.IssueDateTime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.IssuedUnit.PhysicianName), TABLES.IssuedUnit.PhysicianName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.IssuedUnit.IssueToLocationName), TABLES.IssuedUnit.IssueToLocationName);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TreatingSpecialty.TreatingSpecialtyName), TABLES.TreatingSpecialty.TreatingSpecialtyName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitStatus.PresumedTransfusedIndicator), TABLES.BloodUnitStatus.PresumedTransfusedIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusionComment.PatientTransfusionCommentText), TABLES.PatientTransfusionComment.PatientTransfusionCommentText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTransfusion.PatientTransfusionGuid), TABLES.PatientTransfusion.PatientTransfusionGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.ComponentClass.ComponentClassShortName), TABLES.ComponentClass.ComponentClassShortName);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.ComponentClass.ComponentClassName), TABLES.ComponentClass.ComponentClassName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientMiddleName), TABLES.Patient.PatientMiddleName);

		}

		[Test]
		public void GetTransfusionReport_Fail()
		{
			string filter = "";
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now.AddYears(1);

			DataTable dtTest = DAL.PatientTransfusion.GetTransfusionReport(startDate, endDate, filter, "SHNARF",1);

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

	}
}
#endif
